import { NextRequest, NextResponse } from 'next/server';
import connectDB from '@/lib/firebase';
import { imagesCollection } from '@/lib/firestore-collections';

// Enable caching for images - cache for 1 year, revalidate weekly
export const dynamic = 'force-dynamic';
export const revalidate = 604800; // 7 days in seconds

export async function GET(
  request: NextRequest,
  { params }: { params: Promise<{ filename: string }> }
) {
  try {
    const { filename } = await params;
    
    if (!filename) {
      return new NextResponse('Filename is required', { status: 400 });
    }
    
    const db = await connectDB();
    
    // Try to find image in Firestore
    const imageQuery = await imagesCollection()
      .where('filename', '==', filename)
      .limit(1)
      .get();
    
    if (imageQuery.empty) {
      // Return a placeholder image instead of JSON error for better Next.js Image compatibility
      const placeholderSvg = `<svg width="400" height="300" xmlns="http://www.w3.org/2000/svg">
        <rect width="400" height="300" fill="#e5e7eb"/>
        <text x="50%" y="50%" font-family="Arial" font-size="16" fill="#6b7280" text-anchor="middle" dominant-baseline="middle">Image not found</text>
      </svg>`;
      
      return new NextResponse(placeholderSvg, {
        status: 404,
        headers: {
          'Content-Type': 'image/svg+xml',
          'Cache-Control': 'no-store, no-cache, must-revalidate',
        },
      });
    }

    const imageData = imageQuery.docs[0].data();
    
    // If image is stored externally (hosting service URL), redirect to it
    if (imageData.url && (imageData.url.startsWith('http://') || imageData.url.startsWith('https://'))) {
      return NextResponse.redirect(imageData.url, 302);
    }
    
    // If image is stored as data URL, decode and serve it
    if (imageData.url && imageData.url.startsWith('data:')) {
      try {
        const base64Match = imageData.url.match(/^data:([^;]+);base64,(.+)$/);
        if (base64Match) {
          const mimeType = base64Match[1];
          const base64Data = base64Match[2];
          const buffer = Buffer.from(base64Data, 'base64');
          
          return new NextResponse(buffer, {
            headers: {
              'Content-Type': mimeType || imageData.mimeType || 'image/png',
              'Content-Length': buffer.length.toString(),
              'Cache-Control': 'public, max-age=31536000, s-maxage=31536000, immutable, stale-while-revalidate=86400',
              'Access-Control-Allow-Origin': '*',
              'X-Content-Type-Options': 'nosniff',
            },
          });
        }
      } catch (dataUrlError) {
        console.error('Error decoding data URL:', dataUrlError);
      }
    }
    
    // Fallback: Try to fetch from old storageUrl if exists
    if (imageData.storageUrl) {
      try {
        const response = await fetch(imageData.storageUrl);
        if (response.ok) {
          const arrayBuffer = await response.arrayBuffer();
          const buffer = Buffer.from(arrayBuffer);
          
          return new NextResponse(buffer, {
            headers: {
              'Content-Type': imageData.mimeType || 'image/png',
              'Content-Length': buffer.length.toString(),
              'Cache-Control': 'public, max-age=31536000, s-maxage=31536000, immutable, stale-while-revalidate=86400',
              'Access-Control-Allow-Origin': '*',
              'X-Content-Type-Options': 'nosniff',
            },
          });
        }
      } catch (fetchError) {
        console.error('Error fetching from storageUrl:', fetchError);
      }
    }
    
    // If no valid image URL found, return placeholder
    throw new Error('Image URL not found or invalid');
  } catch (error) {
    console.error('Error fetching image:', error);
    // Return a placeholder image on error
    const errorSvg = `<svg width="400" height="300" xmlns="http://www.w3.org/2000/svg">
      <rect width="400" height="300" fill="#fee2e2"/>
      <text x="50%" y="50%" font-family="Arial" font-size="16" fill="#dc2626" text-anchor="middle" dominant-baseline="middle">Error loading image</text>
    </svg>`;
    
    return new NextResponse(errorSvg, {
      status: 500,
      headers: {
        'Content-Type': 'image/svg+xml',
        'Cache-Control': 'no-store, no-cache, must-revalidate',
      },
    });
  }
}

